// Enhanced SkillEnterprise Class with better organization, error handling, and features
class SkillEnterprise {
    constructor() {
        this.API_ENDPOINT = 'https://techhub.com.ng/skent/src/ServerApi/';
        this.state = {
            user: null,
            isAdmin: false,
            currentCourse: null,
            currentVideo: 0,
            autoplay: false,
            loadingState: false
        };
        
        // Configuration
        this.config = {
            videoCacheSize: 5, // Number of videos to preload
            autoSaveInterval: 30000, // Auto-save progress every 30 seconds
            notificationDuration: 3000 // Duration for toast notifications
        };

        this.initializeApp();
    }

    async initializeApp() {
        this.initializeEventListeners();
        this.initializeToasts();
        await this.checkSession();
        this.initializeTheme();
    }

    initializeEventListeners() {
        // Auth forms
        $('#loginForm').on('submit', (e) => this.handleLogin(e));
        $('#registerForm').on('submit', (e) => this.handleRegister(e));
        
        // Video controls
        $('#prevVideo').on('click', () => this.previousVideo());
        $('#nextVideo').on('click', () => this.nextVideo());
        $('#autoplay').on('change', (e) => this.toggleAutoplay(e));
        
        // Admin controls
        $('#addCourse').on('click', () => this.showAddCourseModal());
        
        // Navigation
        $('.nav-link').on('click', (e) => this.handleNavigation(e));
        
        // Global error handler
        window.onerror = (msg, url, line) => {
            this.showNotification('error', `An error occurred: ${msg}`);
            return false;
        };

        // Handle video player events
        this.initializeVideoPlayerEvents();
    }

    initializeToasts() {
        // Create toast container
        const toastContainer = $('<div>')
            .addClass('toast-container position-fixed bottom-0 end-0 p-3')
            .appendTo('body');

        this.toastContainer = toastContainer;
    }

    showNotification(type, message) {
        const toast = $(`
            <div class="toast" role="alert">
                <div class="toast-header bg-${type}">
                    <strong class="me-auto text-white">${type.charAt(0).toUpperCase() + type.slice(1)}</strong>
                    <button type="button" class="btn-close" data-bs-dismiss="toast"></button>
                </div>
                <div class="toast-body">${message}</div>
            </div>
        `);

        this.toastContainer.append(toast);
        const bsToast = new bootstrap.Toast(toast[0], {
            delay: this.config.notificationDuration
        });
        bsToast.show();

        toast.on('hidden.bs.toast', () => toast.remove());
    }

    async apiRequest(serverAction, method = 'POST', data = null) {
        this.setState({ loadingState: true });

        try {
            const options = {
                url: this.API_ENDPOINT,
                method: method,
                data: { serveraction: serverAction },
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'Accept': 'application/json'
                }
            };

            if (data) {
                console.log(data);
                
                if (method === 'GET') {
                    options.data = { ...options.data, ...data };
                } else {
                    options.data = JSON.stringify({ ...data, serveraction: serverAction });
                    options.contentType = 'application/json';
                }
            }

            const response = await $.ajax(options);
            
            if (!response.success) {
                throw new Error(response.message || 'Operation failed');
            }
            
            return response;
        } catch (error) {
            const errorMessage = error.responseJSON?.message || error.message || 'Request failed';
            this.showNotification('error', errorMessage);
            throw new Error(errorMessage);
        } finally {
            this.setState({ loadingState: false });
        }
    }

    setState(newState) {
        this.state = { ...this.state, ...newState };
        this.updateUI();
    }

    updateUI() {
        // Update loading state
        if (this.state.loadingState) {
            $('#loadingSpinner').show();
        } else {
            $('#loadingSpinner').hide();
        }

        // Update admin UI elements
        $('.admin-only').toggle(this.state.isAdmin);

        // Update course progress
        if (this.state.currentCourse) {
            this.updateProgressUI();
        }
    }

    initializeTheme() {
        const savedTheme = localStorage.getItem('theme') || 'light';
        this.setTheme(savedTheme);

        // Add theme toggle button
        const themeToggle = $(`
            <button class="btn btn-outline-secondary position-fixed bottom-0 start-0 m-3">
                <i class="fas fa-moon"></i>
            </button>
        `).appendTo('body');

        themeToggle.on('click', () => {
            const newTheme = $('body').hasClass('dark-theme') ? 'light' : 'dark';
            this.setTheme(newTheme);
        });
    }

    setTheme(theme) {
        $('body').toggleClass('dark-theme', theme === 'dark');
        localStorage.setItem('theme', theme);
    }


    async checkSession() {
        try {
            const response = await this.apiRequest('checkSession', 'GET');
            if (response.success && response.data.user) {
                this.setState({
                    user: response.data.user,
                    isAdmin: response.data.user.role !== 'student'
                });
                this.initializeDashboard();
            }
        } catch (error) {
            console.error('Session check failed:', error);
        }
    }

    initializeTheme() {
        const savedTheme = localStorage.getItem('theme') || 'light';
        this.setTheme(savedTheme);

        // Add theme toggle button
        const themeToggle = $(`
            <button class="btn btn-outline-secondary position-fixed bottom-0 start-0 m-3">
                <i class="fas fa-moon"></i>
            </button>
        `).appendTo('body');

        themeToggle.on('click', () => {
            const newTheme = $('body').hasClass('dark-theme') ? 'light' : 'dark';
            this.setTheme(newTheme);
        });
    }

    setTheme(theme) {
        $('body').toggleClass('dark-theme', theme === 'dark');
        localStorage.setItem('theme', theme);
    }

    async handleLogin(e) {
        e.preventDefault();
        const formData = new FormData(e.target);
        
        try {
            const response = await this.apiRequest('login', 'POST', {
                email: formData.get('email'),
                password: formData.get('password')
            });
            
            if (response.success) {
                this.setState({
                    user: response.data.user,
                    isAdmin: response.data.user.role !== 'student'
                });
                
                $('#authModal').modal('hide');
                this.showNotification('success', 'Welcome back!');
                this.initializeDashboard();
            }
        } catch (error) {
            // Error handling done in apiRequest
        }
    }

    initializeVideoPlayerEvents() {
        // Listen for video ended event
        $('#ytPlayer').on('ended', () => {
            if (this.state.autoplay) {
                this.nextVideo();
            }
        });

        // Auto-save progress periodically
        setInterval(() => {
            if (this.state.currentCourse && this.state.user) {
                this.saveProgress();
            }
        }, this.config.autoSaveInterval);
    }

    updateProgressUI() {
        const totalVideos = this.state.currentCourse.videos.length;
        const completedVideos = this.state.currentCourse.videos.filter(v => v.completed).length;
        const progressPercentage = (completedVideos / totalVideos) * 100;

        $('.progress-bar').css('width', `${progressPercentage}%`);
        $('#progressText').text(`${completedVideos}/${totalVideos} completed (${progressPercentage.toFixed(1)}%)`);
    }

    // Additional methods for course management
    async duplicateCourse(courseId) {
        try {
            const response = await this.apiRequest('duplicateCourse', 'POST', { courseId });
            this.showNotification('success', 'Course duplicated successfully');
            this.loadAdminDashboard();
        } catch (error) {
            // Error handling done in apiRequest
        }
    }

    async archiveCourse(courseId) {
        if (!confirm('Are you sure you want to archive this course?')) return;

        try {
            const response = await this.apiRequest('archiveCourse', 'POST', { courseId });
            this.showNotification('success', 'Course archived successfully');
            this.loadAdminDashboard();
        } catch (error) {
            // Error handling done in apiRequest
        }
    }

    // Enhanced video player methods
    async preloadVideos() {
        const currentIndex = this.state.currentVideo;
        const videosToPreload = this.state.currentCourse.videos
            .slice(currentIndex + 1, currentIndex + 1 + this.config.videoCacheSize);

        for (const video of videosToPreload) {
            const preloadLink = document.createElement('link');
            preloadLink.rel = 'preload';
            preloadLink.as = 'video';
            preloadLink.href = `https://www.youtube.com/embed/${video.youtube_id}`;
            document.head.appendChild(preloadLink);
        }
    }
}

// Initialize the application with error boundary
$(document).ready(() => {
    try {
        window.skillEnterprise = new SkillEnterprise();
    } catch (error) {
        console.error('Failed to initialize application:', error);
        $('body').html(`
            <div class="alert alert-danger m-3">
                Failed to initialize application. Please refresh the page or contact support.
            </div>
        `);
    }
});